/** @file   storyviewer.cpp
 * @brief   Implementation of StoryViewer - class.
 * @version $Revision: 1.3 $
 * @author  Tomi Lamminsaari
 */

#include "storyviewer.h"
#include "display.h"
#include "sprite.h"
#include "consts.h"

using std::string;
using namespace eng2d;

namespace WeWantWar {


/** Upper Y coordinate of the storyview area */
const float KStoryRectUpperY = 320;

/** Lower Y coordinate of the storyview area */
const float KStoryRectLowerY = 480;


/** Story text scrolling speed (pixels in second) */
const float KStoryScrollSpeed = 0.25;


/** Constructor
 */
StoryViewer::StoryViewer()
{
}



/** Destructor
 */
StoryViewer::~StoryViewer()
{
}



/** Shows the level story
 */
int StoryViewer::showStory( const string& aBackground,
                            const string& aStoryFile )
{
  BITMAP* pBackground = load_bitmap( aBackground.c_str(), 0 );
  BITMAP* pStory = load_bitmap( aStoryFile.c_str(), 0 );
  
  if ( pBackground == 0 || pStory == 0 ) {
    return -1;
  }
  
  this->storyScroll( pBackground, pStory );
  
  destroy_bitmap( pBackground );
  destroy_bitmap( pStory );
  return 0;
}


/** Does the story scroll
 */
void StoryViewer::storyScroll( BITMAP* aBackground, BITMAP* aStory )
{

  float posY = KStoryRectLowerY;
  float scrollSpeed = KStoryScrollSpeed;
  
  // Load and play the background sample.
  SAMPLE* pSamp = load_sample( "snd/atmosphere/story.wav" );
  if ( pSamp != 0 ) {
    play_sample( pSamp, 180, 128, 1000, 1 );
  }
  
  while ( key[KEY_ENTER] || key[KEY_SPACE] || key[KEY_ESC] );
  
  //timerCounter = 0;   // found in consts.h
  bool quitNow = false;
  int frameSkip = 0;
  FpsTimer* gameTimer = FpsTimer::getInstance();
  bool needsRedraw = false;
  
  while ( quitNow == false ) {
    if ( key[KEY_ENTER] || key[KEY_SPACE] || key[KEY_ESC] ) {
      quitNow = true;
    }

    frameSkip = 0;
    if ( gameTimer->updatesToGo() > 0 ) {
      do {
        // We scroll the text.
        posY -= scrollSpeed;
        float bottomY = posY + static_cast<float>( aStory->h );
        if ( bottomY < KStoryRectUpperY ) {
          // The lower edge of the story bitmap has reached to upper
          // edge of the scrolling area. Start new scroll.
          posY = KStoryRectLowerY;
        }
        
        gameTimer->updateDone();
        frameSkip += 1;
        if ( frameSkip >= 6 ) {
          gameTimer->reset();
          break;
        }
        
      } while ( gameTimer->updatesToGo() > 0 );
      needsRedraw = true;
    }
    
    // Draw the graphics
    if ( needsRedraw == true ) {
      needsRedraw = false;
      blit( aBackground, Display::buffer, 0,0, 0,0, 640,480 );
    
      set_clip_rect( Display::buffer,
                     0,static_cast<int>(KStoryRectUpperY) + 10,
                     640,480 );
      blit( aStory, Display::buffer, 0,0, 0,static_cast<int>(posY),
            640,aStory->h );
      set_clip_rect( Display::buffer, 0,0, Display::scrWidth(),Display::scrHeight() );
    
      Display::flip();
    }
  }
  
  FpsTimer::releaseInstance( gameTimer );
  // Destroy the sample
  if ( pSamp != 0 ) {
    stop_sample( pSamp );
    destroy_sample( pSamp );
  }
}


} // end of namespace
